import { SanitizerOptions } from '../types';

export const sanitizeText = (text: string, options: SanitizerOptions): string => {
  if (!text) return '';
  
  let result = text;

  // Remove HTML tags
  if (options.removeHtmlTags) {
    result = result.replace(/<[^>]*>/g, '');
  }

  // Convert URLs to clickable links
  if (options.convertUrlsToLinks) {
    result = result.replace(
      /(https?:\/\/[^\s]+)/g,
      '<a href="$1" target="_blank" rel="noopener noreferrer">$1</a>'
    );
  }

  // Remove emojis
  if (options.stripEmojis) {
    result = result.replace(/[\u{1F300}-\u{1F6FF}\u{1F900}-\u{1F9FF}]/gu, '');
  }

  // Remove punctuation
  if (options.removePunctuation) {
    result = result.replace(/[.,\/#!$%\^&\*;:{}=\-_`~()]/g, '');
  }

  // Normalize all newlines to \n
  if (options.normalizeNewlines) {
    result = result.replace(/\r\n|\r/g, '\n');
  }

  // Remove empty lines
  if (options.removeEmptyLines) {
    result = result.split('\n').filter(line => line.trim() !== '').join('\n');
  }

  // Remove leading spaces
  if (options.removeLeadingSpaces) {
    result = result.split('\n').map(line => line.trimStart()).join('\n');
  }

  // Remove trailing spaces
  if (options.removeTrailingSpaces) {
    result = result.split('\n').map(line => line.trimEnd()).join('\n');
  }

  // Remove extra spaces
  if (options.removeExtraSpaces) {
    result = result.replace(/\s+/g, ' ').trim();
  }

  // Capitalize first letter of sentences
  if (options.capitalizeFirstLetter) {
    result = result.replace(
      /(^\w|\.\s+\w)/g,
      letter => letter.toUpperCase()
    );
  }

  return result;
}